/*****************************************************************************
 * ugBASIC - an isomorphic BASIC language compiler for retrocomputers        *
 *****************************************************************************
 * Copyright 2021-2022 Marco Spedaletti (asimov@mclink.it)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *----------------------------------------------------------------------------
 * Concesso in licenza secondo i termini della Licenza Apache, versione 2.0
 * (la "Licenza"); è proibito usare questo file se non in conformità alla
 * Licenza. Una copia della Licenza è disponibile all'indirizzo:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Se non richiesto dalla legislazione vigente o concordato per iscritto,
 * il software distribuito nei termini della Licenza è distribuito
 * "COSÌ COM'È", SENZA GARANZIE O CONDIZIONI DI ALCUN TIPO, esplicite o
 * implicite. Consultare la Licenza per il testo specifico che regola le
 * autorizzazioni e le limitazioni previste dalla medesima.
 ****************************************************************************/

/****************************************************************************
 * INCLUDE SECTION 
 ****************************************************************************/

#include "../../ugbc.h"

/****************************************************************************
 * CODE SECTION 
 ****************************************************************************/

/**
 * @brief Emit code for <strong>SPRITE(...)</strong>
 * 
 * @param _environment Current calling environment
 * @param _image image to use as SPRITE
 */
/* <usermanual>
@keyword SPRITE

@target c64
</usermanual> */
Variable * sprite_init( Environment * _environment, char * _image, char * _sprite, int _flags ) {

    Variable * index;

    if ( _sprite ) {

        index = variable_retrieve_or_define( _environment, _sprite, VT_SPRITE, 0 );

    } else {

        Variable * spriteCount = variable_retrieve( _environment, "SPRITECOUNT" );
        index = variable_temporary( _environment, VT_SPRITE, "(sprite index)" );
        variable_move_naked( _environment, spriteCount->name, index->name );
        cpu_inc( _environment, spriteCount->realName );

    }

    Variable * image = variable_retrieve( _environment, _image );

    Variable * realImage = sprite_converter( _environment, image->originalBitmap, image->originalWidth, image->originalHeight, NULL, _flags );

    if ( _flags & SPRITE_FLAG_MULTICOLOR) {
        sprite_multicolor_var( _environment, index->name );
    } else {
        sprite_monocolor_var( _environment, index->name );
    }
    
    vic2_sprite_data_from( _environment, index->name, realImage->name );

    if ( _flags & SPRITE_FLAG_EXPAND_VERTICAL) {
        sprite_expand_horizontal_var( _environment, index->name );
    } else {
        sprite_compress_horizontal_var( _environment, index->name );
    }

    if ( _flags & SPRITE_FLAG_EXPAND_HORIZONTAL) {
        sprite_expand_vertical_var( _environment, index->name );
    } else {
        sprite_compress_vertical_var( _environment, index->name );
    }

    return index;

}
